<?php
namespace common\models;

use Yii;
use yii\base\NotSupportedException;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;
use yii\helpers\ArrayHelper;
/**
 * User model
 *
 * @property integer $id
 * @property string $username
 * @property string $password_hash
 * @property string $password_reset_token
 * @property string $email
 * @property string $auth_key
 * @property integer $status
 * @property integer $created_at
 * @property integer $updated_at
 * @property string $password write-only password
 */
class User extends ActiveRecord implements IdentityInterface
{
    const STATUS_DELETED = 0;
    const STATUS_ACTIVE = 10;
    public $new_password;
    public $new_password_confirm;
    public $old_password;
    public $old_pss;
    public $images;


    /*public $id;
    public $password;
    public $authKey;
    public $accessToken;*/

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return '{{%user}}';
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            TimestampBehavior::className(),
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [

            ['image', 'image', 'extensions'=>'jpg, png'],
            [['first_name', 'last_name', 'description', 'AdresseBundesland', 'AdresseOrt', 'AdresseStreat', 'AdressePLZ','AdresseStiege', 'active', 'Telefonnummer', 'send_mail', 'Betreiber', 'email', 'username', 'stamp', 'email2', 'images', 'BetreiberPdf'], 'safe'],
            ['status', 'default', 'value' => self::STATUS_ACTIVE],
            [['username', 'Betreiber', 'role_id'], 'required'],
            ['status', 'in', 'range' => [self::STATUS_ACTIVE, self::STATUS_DELETED]],
            [['Betreiber', 'AdresseOrt', 'AdresseStreat', 'AdressePLZ', 'Telefonnummer'], 'required', 'on' => 'requiredName'],

            [['first_name', 'last_name'], 'required', "message"=>"Pflichtfeld", 'on' => 'requiredNameCoworker'],
            [['Telefonnummer'], 'required', "message"=>"Bitte geben Sie eine gültige Telefonnummer an.", 'on' => 'requiredNameCoworker'],
            [['Telefonnummer'], 'string','min'=>6, 'max' => 200,'tooShort'=>'Bitte geben Sie eine gültige Telefonnummer an.' , 'tooLong' => 'Bitte geben Sie eine gültige Telefonnummer an.'],
            [['email'], 'required', 'on' => 'requiredNameCoworker'],
            [['new_password', 'new_password_confirm', 'old_password'], 'required', "message"=> "Bitte geben Sie Ihr Passwort ein.", 'on' => 'changePwd'],
            [['old_password', 'new_password', 'new_password_confirm'], 'string', 'min' => 8, "tooShort" => "8 oder mehr Zeichen mit einer Mischung aus Buchstaben, Ziffern und Symbolen verwenden"],
            ['new_password_confirm', 'compare', 'compareAttribute'=>'new_password', "message" => "Die Passwörter stimmen nicht überein. Versuchen Sie es erneut. ", 'on'=>'changePwd'],
            [['new_password', 'new_password_confirm'], 'required', 'on' => 'changePswAdmin'],
            ['new_password_confirm', 'compare', 'compareAttribute'=>'new_password', 'on'=>'changePswAdmin'],
            [['first_name', 'last_name'], 'required', "message"=>"Pflichtfeld", 'on' => 'coworker'],
            [['Telefonnummer'], 'required', "message"=>"Bitte geben Sie eine gültige Telefonnummer an.", 'on' => 'coworker'],
            [['Telefonnummer'], 'string','min'=>6, 'max' => 200,'tooShort'=>'Bitte geben Sie eine gültige Telefonnummer an.' , 'tooLong' => 'Bitte geben Sie eine gültige Telefonnummer an.'],
            [['new_password', 'new_password_confirm'], 'required', "message"=> "Bitte geben Sie Ihr Passwort ein.", 'on' => 'coworker'],
            ['new_password_confirm', 'compare', 'compareAttribute'=>'new_password', "message" => "Die Passwörter stimmen nicht überein. Versuchen Sie es erneut. ", 'on'=>'coworker'],            

        ];
    }

    /**
     * @inheritdoc
     */
    public static function findIdentity($id)
    {
        return static::findOne(['id' => $id, 'status' => self::STATUS_ACTIVE]);
    }

    /**
     * @inheritdoc
     */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        throw new NotSupportedException('"findIdentityByAccessToken" is not implemented.');
    }

    /**
     * Finds user by username
     *
     * @param string $username
     * @return static|null
     */
    public static function findByUsername($username)
    {
        return static::findOne(['username' => $username, 'status' => self::STATUS_ACTIVE]);
    }

    /**
     * Finds user by password reset token
     *
     * @param string $token password reset token
     * @return static|null
     */
    public static function findByPasswordResetToken($token)
    {
        if (!static::isPasswordResetTokenValid($token)) {
            return null;
        }

        return static::findOne([
            'password_reset_token' => $token,
            'status' => self::STATUS_ACTIVE,
        ]);
    }

    /**
     * Finds out if password reset token is valid
     *
     * @param string $token password reset token
     * @return bool
     */
    public static function isPasswordResetTokenValid($token)
    {
        if (empty($token)) {
            return false;
        }

        $timestamp = (int) substr($token, strrpos($token, '_') + 1);
        $expire = Yii::$app->params['user.passwordResetTokenExpire'];
        return $timestamp + $expire >= time();
    }

    /**
     * @inheritdoc
     */
    public function getId()
    {
        return $this->getPrimaryKey();
    }

    /**
     * @inheritdoc
     */
    public function getAuthKey()
    {
        return $this->auth_key;
    }

    /**
     * @inheritdoc
     */
    public function validateAuthKey($authKey)
    {
        return $this->getAuthKey() === $authKey;
    }

    /**
     * Validates password
     *
     * @param string $password password to validate
     * @return bool if password provided is valid for current user
     */
    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    /**
     * Generates password hash from password and sets it to the model
     *
     * @param string $password
     */
    public function setPassword($password)
    {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password);
    }

    /**
     * Generates "remember me" authentication key
     */
    public function generateAuthKey()
    {
        $this->auth_key = Yii::$app->security->generateRandomString();
    }

    /**
     * Generates new password reset token
     */
    public function generatePasswordResetToken()
    {
        $this->password_reset_token = Yii::$app->security->generateRandomString() . '_' . time();
    }

    /**
     * Removes password reset token
     */
    public function removePasswordResetToken()
    {
        $this->password_reset_token = null;
    }


    /**
     * @return \yii\db\ActiveQuery
     */
    public function getRole()
    {
        return $this->hasOne(Role::className(), ['id' => 'role_id']);
    }

    public function fuullName()
    {
        return $this->first_name. $this->last_name ? "{$this->first_name} {$this->last_name}" : "Not Set";
    }


    public function PDFfullName()
    {
        return $this->first_name. $this->last_name ? "{$this->first_name} {$this->last_name}" : "Belmir Hamidovic";
    }

    public function profileImage()
    {
        $s3_env = S3_PATH;
        return $this->image ? $s3_env. "profile/".$this->image : "https://dummyimage.com/180/000000/fff.png&text=Profile+image";
    }


    public function profileStamp()
    {
        $s3_env = S3_PATH;
        return $this->stamp ? $s3_env. "profile/".$this->stamp : "https://freieingenieure.s3.eu-central-1.amazonaws.com/live/image/pecat.jpg";
    }

    public function attributeLabels()
    {
        return [
            'image' => 'Profile image',
            'username' => 'Email',
            'company_id' => 'Company',
            'role_id' => 'User Role',
            'Betreiber' => 'Betreiber',
            'AdresseBundesland' => Yii::t('app', 'Bundesland'),
            'AdresseOrt' => Yii::t('app', 'Ort'),
            'AdresseStreat' => Yii::t('app', 'Strasse'),
            'AdresseStiege' => Yii::t('app', 'Stiege'),
            'AdressePLZ' => Yii::t('app', 'PLZ'),
        ];
    }

    public function currentUser()
    {
        $id = Yii::$app->user->getId();
        if($id)
        {
            $user= User::findOne($id);
            return $user;
        }else {
            $user = false;
        }
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTehnicals()
    {
        return $this->hasMany(Tehnical::className(), ['user_id' => 'id']);
    }

    public function getUserInvitations()
    {
        return $this->hasMany(UserInvitation::className(), ['user_id' => 'id']);
    }

    public function activation_button_toggle()
    {

        $coworkers = Coworker::find()->where(['company_id' => $this->id])->all();
        $covorkers_ids = ArrayHelper::map($coworkers,'id', 'user_id');

        $inactive_workers = User::find()->where(['in', 'id', $covorkers_ids])->andWhere('coworker_active= :coworker_active', [':coworker_active' => false]);


        if ( $inactive_workers->count() > 0) {
            return true;
        }else {
            return false;
        }     
    }

    public function getResponsive()
    {
        return $this->hasMany(Responsive::className(), ['user_id' => 'id'])->one();
    }

    public function getAdminCoworker()
    {
        return $this->hasOne(CoworkerControll::className(), ['id' => 'user_id']);
    }

    public function getCoworekerPermission()
    {
        return $this->hasMany(CoworkerControll::className(), ['user_id' => 'id'])->orderBy(['id'=>SORT_DESC])->one();
    }
}
